/*
 * Decompiled with CFR 0.152.
 */
package de.rcenvironment.core.component.integration.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import de.rcenvironment.core.component.integration.ToolIntegrationContext;
import de.rcenvironment.core.component.integration.ToolIntegrationService;
import de.rcenvironment.core.utils.common.JsonUtils;
import de.rcenvironment.core.utils.common.StringUtils;
import de.rcenvironment.toolkit.modules.concurrency.api.TaskDescription;
import java.io.File;
import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ToolIntegrationFileWatcher
implements Runnable {
    protected static final Log LOGGER = LogFactory.getLog(ToolIntegrationFileWatcher.class);
    private static final int MAX_RETRIES_REGISTER_ON_CREATE = 5;
    private static final int MAX_RETRIES_INTEGRATE_NEW_FILE = 5;
    private static final int SLEEPING_TIME = 50;
    private WatchService watcher;
    private ToolIntegrationContext context;
    private ToolIntegrationService integrationService;
    private Map<WatchKey, Path> registeredKeys;
    private AtomicBoolean isActive = new AtomicBoolean(true);
    private Map<Path, Long> lastModified;
    private Path rootContextPath;
    private ObjectMapper mapper = JsonUtils.getDefaultObjectMapper();
    private CountDownLatch stoppingLatch;

    public ToolIntegrationFileWatcher(ToolIntegrationContext context, ToolIntegrationService integrationService) throws IOException {
        this.watcher = FileSystems.getDefault().newWatchService();
        this.context = context;
        this.integrationService = integrationService;
        this.registeredKeys = new HashMap<WatchKey, Path>();
        this.lastModified = new HashMap<Path, Long>();
        this.rootContextPath = FileSystems.getDefault().getPath(context.getRootPathToToolIntegrationDirectory(), context.getNameOfToolIntegrationDirectory());
    }

    public void registerRecursive(Path path) throws IOException {
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                try {
                    ToolIntegrationFileWatcher.this.register(dir);
                }
                catch (IOException e) {
                    LOGGER.debug((Object)StringUtils.format((String)"Problem registering directory %s to watch service: %s", (Object[])new Object[]{dir, e.getMessage()}));
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public void register(Path dir) throws IOException {
        if (!this.registeredKeys.containsValue(dir)) {
            WatchKey key = dir.register(this.watcher, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY);
            this.registeredKeys.put(key, dir);
        }
    }

    public void unregisterRecursive(Path path) throws IOException {
        LinkedList<WatchKey> remove = new LinkedList<WatchKey>();
        for (WatchKey key : this.registeredKeys.keySet()) {
            if (!this.registeredKeys.get(key).startsWith(path)) continue;
            remove.add(key);
            key.cancel();
        }
        for (WatchKey key : remove) {
            this.registeredKeys.remove(key);
        }
    }

    public void setWatcherActive(boolean value) {
        this.isActive.set(value);
    }

    public void stop() {
        this.stoppingLatch = new CountDownLatch(1);
        try {
            this.watcher.close();
            this.stoppingLatch.await(5L, TimeUnit.SECONDS);
        }
        catch (IOException | InterruptedException e) {
            LOGGER.error((Object)"Error stopping watcher thread:", (Throwable)e);
        }
    }

    @Override
    @TaskDescription(value="Filewatcher for integration files")
    public void run() {
        boolean running = true;
        while (running) {
            WatchKey key = null;
            try {
                if (this.watcher != null) {
                    key = this.watcher.take();
                } else {
                    running = false;
                }
            }
            catch (InterruptedException e) {
                LOGGER.debug((Object)("Got interrupted waiting for watch keys. " + e.getMessage()));
                return;
            }
            catch (ClosedWatchServiceException closedWatchServiceException) {
                running = false;
                LOGGER.debug((Object)("Shut down watcher for context " + this.context.getContextType()));
                continue;
            }
            if (key == null) {
                LOGGER.debug((Object)("Got null WatchKey for FileWatcher of type " + this.context.getContextType()));
                continue;
            }
            Path directory = this.registeredKeys.get(key);
            if (directory == null) {
                LOGGER.debug((Object)StringUtils.format((String)"Got unregistered WatchKey for FileWatcher of type %s", (Object[])new Object[]{this.context.getContextType()}));
                continue;
            }
            for (WatchEvent<?> event : key.pollEvents()) {
                WatchEvent.Kind<?> kind = event.kind();
                WatchEvent<?> ev = event;
                Path name = (Path)ev.context();
                Path child = directory.resolve(name);
                if (kind == StandardWatchEventKinds.OVERFLOW) continue;
                LOGGER.debug((Object)StringUtils.format((String)"Got event %s in context %s for file: %s", (Object[])new Object[]{kind.name(), this.context.getContextType(), child.toString()}));
                if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                    this.handleCreate(child, directory);
                    continue;
                }
                if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                    this.handleDelete(child, directory);
                    continue;
                }
                if (kind != StandardWatchEventKinds.ENTRY_MODIFY) continue;
                this.handleModify(child, directory);
            }
            key.reset();
        }
        this.stoppingLatch.countDown();
    }

    private void handleCreate(Path child, Path directory) {
        boolean registered = false;
        int attempt = 0;
        while (!registered && attempt < 5) {
            try {
                this.registerRecursive(child);
                registered = true;
            }
            catch (IOException x) {
                registered = false;
                LOGGER.error((Object)StringUtils.format((String)"Could not register new path (Tried %s of %s times): %s; Cause: %s", (Object[])new Object[]{++attempt, 5, child.toString(), x}));
                try {
                    Thread.sleep(50L);
                }
                catch (InterruptedException e1) {
                    LOGGER.debug((Object)"Integration watcher sleep interrupted.", (Throwable)e1);
                }
            }
        }
        if (attempt == 5) {
            LOGGER.error((Object)StringUtils.format((String)"Could not register new path after %s tries: %s", (Object[])new Object[]{5, child.toString()}));
        }
        if (this.isActive.get()) {
            if (Files.isDirectory(child, new LinkOption[0])) {
                if (child.getNameCount() == this.rootContextPath.getNameCount() + 1) {
                    File configurationFile = new File(child.toFile(), this.context.getConfigurationFilename());
                    this.integrateFile(configurationFile);
                }
            } else if (Files.isRegularFile(child, new LinkOption[0])) {
                if (child.endsWith("published.conf")) {
                    this.integrationService.updatePublishedComponents(this.context);
                } else if (child.endsWith(this.context.getConfigurationFilename())) {
                    this.integrateFile(child.toFile());
                } else if (child.getName(child.getNameCount() - 2).endsWith("docs")) {
                    File configurationFile = new File(child.getParent().getParent().toFile(), this.context.getConfigurationFilename());
                    this.removeAndReintegrate(child.getParent().getParent().toFile(), configurationFile);
                }
            }
        } else {
            LOGGER.debug((Object)"Did not handle create because watcher is inactive.");
        }
    }

    private void handleDelete(Path child, Path directory) {
        try {
            this.unregisterRecursive(child);
        }
        catch (IOException x) {
            LOGGER.debug((Object)("Could not unregister path: " + child.toString()), (Throwable)x);
        }
        if (this.isActive.get()) {
            if (this.rootContextPath.equals(directory)) {
                if (child.endsWith("published.conf")) {
                    this.integrationService.updatePublishedComponents(this.context);
                } else if (child.getNameCount() == this.rootContextPath.getNameCount() + 1) {
                    this.removeTool(child.toFile());
                }
            } else if (child.getName(child.getNameCount() - 1).endsWith("docs")) {
                File configurationFile = new File(child.getParent().toFile(), this.context.getConfigurationFilename());
                this.removeAndReintegrate(child.getParent().toFile(), configurationFile);
            } else if (child.getName(child.getNameCount() - 2).endsWith("docs")) {
                File configurationFile = new File(child.getParent().getParent().toFile(), this.context.getConfigurationFilename());
                this.removeAndReintegrate(child.getParent().getParent().toFile(), configurationFile);
            }
        } else {
            LOGGER.debug((Object)"Did not handle delete because watcher is inactive.");
        }
    }

    private void handleModify(Path child, Path directory) {
        if (this.isActive.get()) {
            long currentTime = System.currentTimeMillis();
            if (this.lastModified.get(child) == null) {
                this.modify(child, directory);
            } else if (currentTime - this.lastModified.get(child) > 200L) {
                this.modify(child, directory);
            } else {
                LOGGER.debug((Object)("Skipped modify event because of too frequent calls for " + child.getFileName()));
            }
            this.lastModified.put(child, currentTime);
        }
    }

    private void modify(Path child, Path directory) {
        File configurationFile;
        try {
            Thread.sleep(150L);
        }
        catch (InterruptedException interruptedException) {
            LOGGER.debug((Object)"Sleeping for modify event interrupted");
        }
        if (child.endsWith("published.conf")) {
            this.integrationService.updatePublishedComponents(this.context);
        }
        if (directory.getNameCount() == this.rootContextPath.getNameCount() + 1) {
            configurationFile = new File(directory.toFile(), this.context.getConfigurationFilename());
            this.removeAndReintegrate(directory.toFile(), configurationFile);
        }
        if (directory.getName(directory.getNameCount() - 1).endsWith("docs")) {
            configurationFile = new File(directory.getParent().toFile(), this.context.getConfigurationFilename());
            this.removeAndReintegrate(directory.getParent().toFile(), configurationFile);
        }
    }

    private void removeAndReintegrate(File toRemove, File toIntegrate) {
        LOGGER.debug((Object)("Reload tool configuration for " + toRemove.getName()));
        this.removeTool(toRemove);
        this.integrateFile(toIntegrate);
    }

    private void removeTool(File toolDir) {
        String toolName = this.integrationService.getToolNameToPath(toolDir.getAbsolutePath());
        if (toolName != null) {
            this.integrationService.removeTool(toolName, this.context);
        }
    }

    private void integrateFile(File newConfiguration) {
        boolean read = false;
        int attempt = 0;
        while (!read && attempt < 5) {
            try {
                if (newConfiguration.exists() && newConfiguration.getAbsolutePath().endsWith(".json")) {
                    Map configuration = (Map)this.mapper.readValue(newConfiguration, new HashMap().getClass());
                    this.integrationService.integrateTool(configuration, this.context);
                    this.integrationService.putToolNameToPath((String)configuration.get("toolName"), newConfiguration.getParentFile());
                    read = true;
                    continue;
                }
                LOGGER.debug((Object)StringUtils.format((String)"Configuration file does not exist or is no json file: %s", (Object[])new Object[]{newConfiguration.getAbsolutePath()}));
                read = true;
            }
            catch (IOException e) {
                read = false;
                LOGGER.error((Object)StringUtils.format((String)"Could not read tool configuration (Tried %s of %s times)", (Object[])new Object[]{++attempt, 5}), (Throwable)e);
                try {
                    Thread.sleep(50L);
                }
                catch (InterruptedException interruptedException) {
                    LOGGER.error((Object)"Integration watcher sleep interrupted.");
                }
            }
        }
        if (attempt == 5) {
            LOGGER.error((Object)StringUtils.format((String)"Could not read tool configuration after %s times. Path: %s", (Object[])new Object[]{5, newConfiguration.getAbsolutePath()}));
        }
    }

    public Map<WatchKey, Path> getRegisteredPaths() {
        return this.registeredKeys;
    }
}

